/****************************************************************

     Turbo Prolog Toolbox
     (C) Copyright 1987 Borland International.

  Transmission protocol : (A true subset of the XMODEM PROTOCOL)

 Memonics 	ASCII value	Description
   ACK		    6		acknowledgment
   NAK		   21		Negative acknowledgment
   SOH		    1		Start of data character
   EOT		    4		End of transmisssion


Packet format :
       1                  2                  3-130             131
   Packet No,   Complement to Packet No,   Data (128) ...,   Check sum


Protocol:   

   TRANSMITTER					RECEIVER
   ===========                                  ========
   
						Send NAK (repeat until SOH)
   Send SOH
   Send packet
   						Everything is ok:
   						Send NAK
   						Everything is not ok:
   						Send ACK
   	This will be repeated until transmission is accepted.
   	When all packages are transmitted the transmitter will:
   Send EOT

****************************************************************/


project "xxmodem"

Domains
  FILE = sp; dp

DATABASE
  last_char(Char)
  retrans_coun(Integer)

include "tdoms.pro"
include "comglobs.pro"
include "tpreds.pro"
include "menu.pro"

Domains
  Package	=	package(Char,Char,DataL)
  DataL		=	Char*

Predicates
  run
  decide(Integer)
  send_file(Char)
  receive_file(Char)
  reset_last_char
  read_list(Integer,DataL)
  mkList(Integer, DataL)
  write_list(DataL)

  /* Transmission predicates */
  Send_Package(Package)
  send_data(Integer,DataL,Char)
  reset_retransmit_counter
  increment_retransmit_counter  
  send_and_wait(Char,Char,Integer)
  
  receive_package(Char,Char,Package)
  wait_trans_ready
  receive_data(Integer,Char,DataL)
  Rxch_RS232_delay(Integer,Char)  /* receive with a time out */
  check_notEOT

  check_next_char(Char)
  ignore_until_received(Char)
  test_for_nak

  mess(String,Char)
  assert_char(Char)
  headline(Char)
  opd_headline(Integer,Char,Char)


GOAL
   run.
   
CLAUSES
   run:-
  	makewindow(3, 81,21," Package window ", 0,44,12,36),
  	makewindow(1, 42,36," Message window ", 0,0,12,44),
	makewindow(2, 63,5," Transmission window ", 12,0,12,80),

	PortNo		= 	1,	/* COM1 */
	InputBufSize	=	256,	/* Size of input buffer */
	OutputBufSize	=	256,	/* Size of output buffer */
	BaudRate	=	7,	/* 9600 bits per second	*/
	Parity		=	0,	/* No parity		*/
	WordLength	=	3,	/* Eight data bits	*/
	StopBits	=	0,	/* One stop bits	*/
  	Protocol	= 	2,	/* Xon/Xoff can not be used */
  	openRs232(PortNo, InputBufSize, OutputBufSize, BaudRate, Parity,
  		  WordLength, StopBits, Protocol),
	repeat,
	menu(10,20,64,23,
	     ["Transmit Data.TRS",
  	      "Receive  Data.RCV",
     	      "Quit"],
              "Choose an option",0,Choice),
	decide(Choice),
	fail.
   run:- closeRS232(1).

  /* Transmit a file using Packaging */
  decide(1):-
  	openread(sp,"DATA.TRS"), readdevice(sp),
  	send_file('\001'), !, closefile(sp).
  decide(1):-closefile(sp).
  
  /* Receive a file using Packaging */
  decide(2):-
  	openwrite(dp,"DATA.RCV"), writedevice(dp),
  	receive_file('\001'), !, closefile(dp).
  decide(2):-closefile(dp).
  
  /* Quit */
  decide(3):-closeRS232(1), exit.

/*************************************************************
	Transmit a file
**************************************************************/
  
  send_file(Pno):-
  	read_list(0,DataL),!, char_int(PNO,V), V2=-V, char_int(CPNO,V2),
  	send_package(package(PNO,CPNO,DataL)),
  	V3 = V+1, char_int(PNO2,V3),
  	send_file(PNO2).

  send_file(_):-
  	/* read_list  (read characters from a file) failed */
  	/* Everything is ok - send an EOT and wait for ACK */
  	mess("Send EOT and wait on ACK",' '),
  	send_and_wait('\04','\06',5).


/*************************************************************
	Receive a file
**************************************************************/
  
  receive_file(PNO):-
  	char_int(PNO,V), V2=-V, char_int(CPNO,V2),
  	reset_last_char,
  	receive_package(PNO,CPNO,package(_,_,DATAL)),!,
  	write_list(DATAL),
  	V3 = V+1, char_int(PNO2,V3),
  	receive_file(PNO2).

  receive_file(_).
  /* received_package failed, that means a EOT has been received
     or the transmission is time outed
  */
  
  reset_last_char:-not(last_char(_)),!.
  reset_last_char:-retract(last_char(_)),!.


/*************************************************************
	Support predicates for send_file and receive_file
	read_list	read if it is possible 128 bytes from
			file and convert it to a list.
	write_list	write a list of characters to the current
			output device
**************************************************************/
  
  read_list(128,[]):-!.
  read_list(I,[H|T]):-	readchar(H), !, I2=I+1, read_list(I2,T).
  read_list(I,L):-I>0, Len=128-I, mkList(Len,L).
  
  mkList(0,[]):-!.
  mkList(I,['\026'|T]):-I2 = I - 1, mkList(I2,T).

  write_list([]):-!.
  write_list([H|T]):-write(H), write_list(T).


/*************************************************************
	Transmit a Package
**************************************************************/

  Send_Package(package(PNO,CPNO,DataL)):-
  	headline(PNO),
  	/* wait for a NAK from the receiver */
  	mess("Wait for NAK ...",' '),
  	ignore_until_received('\021'), DelInbuf_RS232(1),
  	/* Send SOH */
  	Txch_RS232(1,'\001'),
  	mess("Send SOH",' '), ticks(20), DelInbuf_RS232(1),
  	/* Send Package number and the complements */
  	mess("Transmit PNO ",PNO),
  	Txch_RS232(1,Pno), ticks(1), test_for_nak,
  	mess("Transmit CPNO ",CPNO),
  	Txch_RS232(1,CPNO), ticks(1), test_for_nak,
  	mess("Transmit Data ...",' '),
  	send_data(0,DataL,'\0'),		/* Send data and checksum */
  	mess("Wait for ACK ...",' '),
  	check_next_char('\006'),  	/* Wait for acknowledgment */
  	reset_retransmit_counter,!.

  Send_Package(package(PNO,CPNO,DataL)):-
  	/* Something is going wrong */
  	/* We will retransmit the same package at most 5 times */
  	increment_retransmit_counter,
  	mess("Retransmitting Package ",PNO),
  	DelInBuf_RS232(1), DelOutBuf_RS232(1),
  	Send_Package(package(PNO,CPNO,DataL)).


/*************************************************************
	Retransmission predicates used by send_Package
**************************************************************/

  reset_retransmit_counter:-not(retrans_coun(_)), !.
  reset_retransmit_counter:-retract(retrans_coun(_)), !.

  increment_retransmit_counter:-not(retrans_coun(_)),assert(retrans_coun(1)),!.
  increment_retransmit_counter:-
	retract(retrans_coun(I)), I<5, I2=I+1, !, assert(retrans_coun(I2)).
  increment_retransmit_counter:-
  	DelInBuf_RS232(1), DelOutBuf_RS232(1),
	mess("\nError transmitting package, Transmission ABORTED\n",' '),
	fail.

  send_and_wait(CH1,CH2,_):-
	Txch_RS232(1,CH1),Rxch_RS232_delay(10,CH),CH=CH2,!.
  send_and_wait(_,CH2,_):-
	DelInbuf_RS232(1), Rxch_RS232_delay(10,CH),CH=CH2.


/*************************************************************
	Receive a Package (128 characters)
**************************************************************/

  receive_package(PNO,CPNO,Package):-
	headline(PNO),
	mess("Continue send NAK's and wait for SOH ...",' '),
	wait_trans_ready,	/* send NAK until a SOH is received */
	mess("Wait for package number PNO= ",PNO),
	check_next_char(PNO),	/* Check for correct package number */
	mess("Wait for complement number CPNO= ",CPNO),
	check_next_char(CPNO),	/* and its complement		    */
	mess("Receive DATA ...",' '),
	receive_data(0,'\0',DataL),!, /* receive data and checksum  */
	/* Everything is all right - Send a acknowledgment */
	mess("Data ok - send ACK",' '),
	Txch_RS232(1,'\006'),
	Package = package(PNO,CPNO,DataL).

  receive_package(PNO,CPNO,Package):-
	/* if wait_trans_ready failed then check for receipts of EOT */
	mess("Transmission error or receiption of EOT",' '),
	check_notEOT,
	/* Transmission of current package crashed - send a NAK */
	DelInBuf_RS232(1), DelOutBuf_RS232(1),
	mess("It was a transmission error while receiving package PNO= ",PNO),
	mess("Send NAK because of error in transmission",' '),
	Txch_RS232(1,'\021'),Ticks(10), DelInBuf_RS232(1),
	receive_package(PNO,CPNO,Package). /* Try again */

  check_notEOT:-not(last_char(_)),!. /* last_char is updated by wait_trans_ready */
  check_notEOT:-last_char(CH),CH<>'\004', retract(last_char(CH)),!.

  assert_char(CH):-retract(last_char(_)),assert(last_char(CH)),!.
  assert_char(CH):-assert(last_char(CH)),!.

  wait_trans_ready:-
	/* Send NAK until receiption of SOH */
	Txch_RS232(1,'\021'), ticks(10),
	/* If a character is received, it should be SOH */
	Rxch_RS232(1,CH), assert_char(CH),
	CH='\001',!.
  wait_trans_ready:-
	check_notEOT,!, wait_trans_ready.
  wait_trans_ready:-
	mess("EOT received - Send ACK",' '),
	Txch_RS232(1,'\006'),fail. /*Send ACK after receive EOT*/


/*************************************************************
	Receive a data block (128 characters)
	Fails if a character is not received in
	the specified time out period.
	
	Data and the corresponding no in the package will be
	echoed to the screen.
**************************************************************/

  receive_data(128,CheckSum,[]):-!,check_next_char(CheckSum).
  receive_data(I,Csum1,[CH|T]):-
	Rxch_RS232_delay(50,CH),
	/* Compute the checksum on the fly */
	char_int(Csum1,V1), char_int(Ch,V2),
	V3 = V1 + V2, char_int(Csum2,V3),
	I2 = I+1, opd_headline(I2,Csum2,CH),
	receive_data(I2,Csum2,T).

  /* Receive characters from COM1 with a 5 seconds time out period */
  Rxch_RS232_delay(_,CH):-Rxch_RS232(1,CH),!.
  Rxch_RS232_delay(I,CH):-I>0, !,I2=I-1,Ticks(5),Rxch_RS232_delay(I2,CH).


/*************************************************************
	Transmit a data block

	Fails if the receiver sends a NAK.
	
	Data will be echoed to the screen.
**************************************************************/

  send_data(_,[],CheckSum):-!,Txch_RS232(1,CheckSum).
  send_data(I,[H|T],Csum1):-
  	test_for_nak,
  	Txch_RS232(1,H),
  	char_int(H,V), char_int(Csum1,V1),
  	V2 = V+V1, char_int(Csum2,V2),
  	I2=I+1, opd_headline(I2,Csum2,H),
  	send_data(I2,T,Csum2).

  test_for_nak:-Rxch_RS232(1,CH),CH='\021',!,mess("Received NAK",' '),fail.
  test_for_nak.


/*************************************************************

	Miscellanous Predicates

**************************************************************/

  ignore_until_received(CH):-check_next_char(CH),!.
  ignore_until_received(CH):-ignore_until_received(CH).

  check_next_char(CH):-Rxch_RS232(1,CH1),!,CH1=CH.
  check_next_char(CH):-check_next_char(CH).

  mess(S,CH):-
	writedevice(WD), writedevice(screen),
	shiftwindow(W), gotowindow(1),
	char_int(CH,V), write("\n",S,V),
	gotowindow(W),
	writedevice(WD),
	readdevice(ID),readdevice(keyboard),/*readchar(_),*/ readdevice(ID).
  
  headline(PNO):-
  	writedevice(WD), writedevice(screen),
  	gotowindow(3), cursor(R,_), cursor(R,33),
  	char_int(Pno,V), write("\nPackage NO:",V," Data:"),
  	writedevice(WD).
  
  opd_headline(I,Checksum,CH):-
  	writedevice(WD), writedevice(screen),
  	gotowindow(2), write(CH),
  	gotowindow(3), char_int(Checksum,V),
  	cursor(R,C), write(I," Chksum:",V), cursor(R,C),
  	writedevice(WD).
